<?php
/**
 * SCL-90 核心 API 接口
 * 功能：处理卡密验证、测试数据提交、证据链锁定
 */

// 1. 基础环境引入
require_once 'db_core.php';
require_once 'includes/functions.php';

// 2. 响应头与会话开启
header('Content-Type: application/json; charset=utf-8');
session_start();

// 3. 获取输入数据
$action = $_GET['action'] ?? '';
$rawData = file_get_contents('php://input');
$data = json_decode($rawData, true);

// 4. 路由逻辑
switch ($action) {
    
    /**
     * 🚪 动作：verify (卡密验证)
     * 逻辑：校验卡密有效性，初始化证据链，锁定用户IP
     */
    case 'verify':
        $code = trim($data['code'] ?? '');
        
        if (empty($code)) {
            echo json_encode(['success' => false, 'message' => '请输入卡密']);
            exit;
        }

        try {
            // 检查卡密是否存在
            $stmt = $pdo->prepare("SELECT * FROM cdks WHERE code = ? LIMIT 1");
            $stmt->execute([$code]);
            $cdk = $stmt->fetch();

            if (!$cdk) {
                echo json_encode(['success' => false, 'message' => '卡密无效']);
                exit;
            }

            if ($cdk['status'] == 1) {
                echo json_encode(['success' => false, 'message' => '此卡密已被使用']);
                exit;
            }

            // --- 证据链采集开始 ---
            $ip = getVisitorIP();
            $ua = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
            $now = date('Y-m-d H:i:s');

            $pdo->beginTransaction();

            // 更新卡密状态为“已使用”并锁定使用时间
            $upCdk = $pdo->prepare("UPDATE cdks SET status = 1, used_at = ? WHERE id = ?");
            $upCdk->execute([$now, $cdk['id']]);

            // 创建测试记录，预埋证据（IP/UA/开始时间）
            $resStmt = $pdo->prepare("INSERT INTO test_results (cdk_id, visitor_ip, visitor_ua, start_time) VALUES (?, ?, ?, ?)");
            $resStmt->execute([$cdk['id'], $ip, $ua, $now]);
            
            $recordId = $pdo->lastInsertId();

            $pdo->commit();

            // 写入 Session 权限令牌
            $_SESSION['auth_cdk_id'] = $cdk['id'];
            $_SESSION['test_record_id'] = $recordId;
            $_SESSION['start_timestamp'] = time();

            echo json_encode(['success' => true]);

        } catch (Exception $e) {
            if ($pdo->inTransaction()) $pdo->rollBack();
            echo json_encode(['success' => false, 'message' => '系统错误，请重试']);
        }
        break;

    /**
     * 📊 动作：submit (提交测试结果)
     * 逻辑：校验会话权限，计算用时，存入最终证据数据
     */
    case 'submit':
        // 安全性检查：必须持有验证通过的 Session
        if (!isset($_SESSION['auth_cdk_id']) || !isset($_SESSION['test_record_id'])) {
            echo json_encode(['success' => false, 'message' => '会话已过期，请重新验证']);
            exit;
        }

        $recordId = $_SESSION['test_record_id'];
        $gsi = $data['gsi'] ?? 0;
        $scores = json_encode($data['scores'] ?? []);
        $isComplete = $data['is_complete'] ?? 0;
        $now = date('Y-m-d H:i:s');

        // 计算实际用时（证据链比对）
        // 前端传来的 duration 作为辅助，后端根据 start_timestamp 再次计算防止篡改
        $serverSideDuration = time() - $_SESSION['start_timestamp'];
        $finalDuration = $data['duration'] ?? $serverSideDuration;

        try {
            // 更新测试记录，补全终点证据
            $stmt = $pdo->prepare("UPDATE test_results SET 
                end_time = ?, 
                duration = ?, 
                is_complete = ?, 
                gsi_score = ?, 
                score_json = ? 
                WHERE id = ?");
            
            $stmt->execute([
                $now, 
                $finalDuration, 
                $isComplete, 
                $gsi, 
                $scores, 
                $recordId
            ]);

            // 权限提升：标记已完成测试，允许查看结果页
            $_SESSION['test_finished'] = true;

            echo json_encode(['success' => true]);

        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => '报告生成失败']);
        }
        break;

    /**
     * 🚪 动作：logout (重置会话)
     */
    case 'logout':
        session_destroy();
        echo json_encode(['success' => true]);
        break;

    default:
        echo json_encode(['success' => false, 'message' => 'Invalid API Action']);
        break;
}