<?php
/**
 * SCL-90 深度分析报告 Pro
 * 功能：多维度临床指标、AI 深度画像、管理员卷宗穿透、交付证据链
 */
require_once 'db_core.php';
require_once 'includes/functions.php';

session_start();

// --- 1. 权限与数据调取逻辑 ---
$isAdminView = false;
$recordId = null;

// 判断是否为管理员穿透查看
if (isset($_GET['admin_record_id']) && isset($_SESSION['admin_auth'])) {
    $recordId = intval($_GET['admin_record_id']);
    $isAdminView = true;
} else {
    // 普通用户逻辑：必须完成测试
    if (!isset($_SESSION['test_record_id']) || !isset($_SESSION['test_finished'])) {
        header("Location: index.php");
        exit;
    }
    $recordId = $_SESSION['test_record_id'];
}

try {
    $stmt = $pdo->prepare("
        SELECT r.*, c.code 
        FROM test_results r 
        JOIN cdks c ON r.cdk_id = c.id 
        WHERE r.id = ?
    ");
    $stmt->execute([$recordId]);
    $report = $stmt->fetch();

    if (!$report) die("未找到相关测试卷宗");

    $scores = json_decode($report['score_json'], true);
} catch (Exception $e) {
    die("数据加载异常");
}

/**
 * 🧠 增强型 AI 临床分析引擎
 */
function generateProAIReport($gsi, $scores) {
    $report = [];
    
    // 1. 总体压力状态分析
    if ($gsi < 1.6) {
        $report[] = "🟢 **心理健康底色**：您的总体症状指数（GSI）非常低，显示出极佳的情绪稳定性。在面对生活压力时，您的心理防御机制运作良好，属于健康人群中的佼佼者。";
    } elseif ($gsi < 2.5) {
        $report[] = "🟡 **情绪波动警示**：数据提示您目前处于“亚健康”状态。虽然不至于影响正常生活，但您的心理能量储备正在下降，建议适当减少社交损耗。";
    } else {
        $report[] = "🔴 **高应激状态预警**：您的总均分超过了临床筛查界限。这表明您目前正承受着高强度的心理痛苦，可能伴有明显的失眠、焦虑或躯体不适。建议寻求专业支持。";
    }

    // 2. 因子组合深度深度画像
    // 逻辑：焦躁性抑郁 (ANX + DEP 同时高)
    if ($scores['ANX'] >= 2.2 && $scores['DEP'] >= 2.2) {
        $report[] = "🔍 **深度画像 - 焦躁性抑郁倾向**：您的焦虑与抑郁维度同时偏高。这通常表现为“坐立难安却又动力不足”，既对未来感到担忧，又对当下感到无力。";
    }
    
    // 逻辑：身心联动 (SOM 高)
    if ($scores['SOM'] >= 2.0) {
        $report[] = "🔍 **深度画像 - 躯体化防御**：您的心理压力正在通过生理渠道排解。头痛、肠胃不适或肌肉酸痛可能是“情绪的呼救”，单纯的生理治疗可能效果有限。";
    }

    // 3. 维度针对性建议
    arsort($scores);
    $topDim = key($scores);
    $adviceMap = [
        'SOM' => '建议尝试正念呼吸，将注意力从身体不适转移到呼吸节律上。',
        'OCD' => '尝试练习“推迟焦虑”，给自己的反复思考设定专门的“打卡时间”。',
        'INT' => '在人际中尝试建立“心理边界”，减少对他人的过度共情。',
        'DEP' => '执行“5分钟行动法”，不求结果，只求动起来，打破抑郁循环。',
        'ANX' => '减少咖啡因摄入，增加高强度有氧运动以代谢过量的肾上腺素。',
        'HOS' => '情绪爆发前尝试“冷处理”，离开当前环境 10 分钟。',
        'PHOB' => '采取系统脱敏法，从最微小的不适环境开始逐步适应。',
        'PAR' => '尝试记录“客观事实”而非“主观猜想”，矫正认知偏差。',
        'PSY' => '建议寻找一个安全的私密空间进行深度倾诉，减少孤独感。'
    ];
    
    if ($scores[$topDim] >= 2.0) {
        $report[] = "💡 **精准干预建议**：针对您最突出的项目，" . $adviceMap[$topDim];
    }

    return implode("<br><br>", $report);
}
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1, maximum-scale=1, user-scalable=no, viewport-fit=cover">
    <title>SCL-90 深度报告 Pro</title>
    <link rel="stylesheet" href="assets/style.css">
    <style>
        .pro-header { background: #fff; padding: 40px 20px; text-align: center; border-bottom: 0.5px solid var(--ios-divider); }
        .score-box { display: flex; justify-content: space-around; margin: 20px 0; }
        .score-item { text-align: center; }
        .score-num { font-size: 28px; font-weight: 800; color: var(--ios-blue); }
        .score-tit { font-size: 11px; color: var(--ios-subtext); margin-top: 4px; }
        
        .ai-pro-card { background: #FFFFFF; border-radius: 24px; padding: 24px; margin: 20px; border-left: 6px solid var(--ios-blue); box-shadow: var(--shadow-md); }
        .section-title { font-size: 17px; font-weight: 700; margin: 30px 20px 10px; color: #000; display: flex; align-items: center; gap: 8px; }
        
        .admin-ribbon { background: var(--ios-orange); color: #fff; text-align: center; font-size: 12px; padding: 8px; font-weight: 700; }
    </style>
</head>
<body>

<?php if($isAdminView): ?>
    <div class="admin-ribbon">🛠️ 当前处于管理员调阅模式</div>
<?php endif; ?>

<div class="app-container">
    <div class="pro-header fade-in">
        <div style="font-size: 13px; color: var(--ios-subtext); margin-bottom: 8px;">心理健康深度分析报告</div>
        <div class="score-num" style="font-size: 48px;"><?php echo $report['gsi_score']; ?></div>
        <div class="score-tit">GSI 总症状指数</div>
        
        <div class="score-box">
            <div class="score-item">
                <div class="score-num" style="font-size: 20px; color: #000;"><?php echo formatDuration($report['duration']); ?></div>
                <div class="score-tit">答题用时</div>
            </div>
            <div class="score-item">
                <div class="score-num" style="font-size: 20px; color: #000;">90 / 90</div>
                <div class="score-tit">完成题目</div>
            </div>
        </div>
    </div>

    <div class="section-title"><span>✨</span> AI 智能画像分析</div>
    <div class="ai-pro-card fade-in">
        <div style="line-height: 1.7; font-size: 15px; color: #3A3A3C;">
            <?php echo generateProAIReport($report['gsi_score'], $scores); ?>
        </div>
    </div>

    <div class="section-title"><span>📊</span> 心理因子分布图</div>
    <div class="card" style="margin: 20px; text-align: center;">
        <div id="radarChart"></div>
        <p style="font-size: 11px; color: var(--ios-subtext); margin-top: 10px;">越靠近外圈代表该项症状越明显</p>
    </div>

    <div class="section-title"><span>📋</span> 临床维度明细</div>
    <div class="card" style="margin: 20px; padding: 10px 20px;">
        <?php 
        $dims = [
            'SOM'=>['躯体化','🤕'],'OCD'=>['强迫','🔄'],'INT'=>['人际敏感','🌵'],
            'DEP'=>['抑郁','🌧️'],'ANX'=>['焦虑','😰'],'HOS'=>['敌对','😡'],
            'PHOB'=>['恐怖','👻'],'PAR'=>['偏执','👁️'],'PSY'=>['精神病性','🌀']
        ];
        foreach($scores as $k => $v): 
            if(!isset($dims[$k])) continue;
        ?>
        <div class="res-row">
            <div style="display:flex; align-items:center; gap:10px;">
                <span><?php echo $dims[$k][1]; ?></span>
                <span class="res-name"><?php echo $dims[$k][0]; ?></span>
            </div>
            <div style="font-weight: 700; color: <?php echo $v>=2.5 ? 'var(--ios-red)' : ($v>=2 ? 'var(--ios-orange)' : 'var(--ios-green)'); ?>">
                <?php echo $v; ?>
            </div>
        </div>
        <?php endforeach; ?>
    </div>

    <div class="section-title"><span>🔐</span> 数字化交付证明</div>
    <div class="card" style="margin: 20px; background: #F9F9FB; border: 1px dashed #D2D2D7;">
        <div class="evidence-item"><span>激活卡密</span> <span><?php echo $isAdminView ? $report['code'] : substr($report['code'], 0, 4)."****"; ?></span></div>
        <div class="evidence-item"><span>来访 IP</span> <span><?php echo $report['visitor_ip']; ?></span></div>
        <div class="evidence-item"><span>激活时间</span> <span><?php echo $report['start_time']; ?></span></div>
        <div class="evidence-item"><span>交付时间</span> <span><?php echo $report['end_time']; ?></span></div>
        <div class="evidence-item"><span>交付状态</span> <span style="color:var(--ios-green)">✅ 服务已完整消耗</span></div>
    </div>

    <div style="padding: 0 20px 40px;">
        <button class="btn-primary" onclick="window.print()">保存 PDF 报告</button>
        <?php if($isAdminView): ?>
            <button class="btn-sec" style="margin-top:12px; width:100%;" onclick="window.close()">关闭预览</button>
        <?php else: ?>
            <button class="btn-sec" style="margin-top:12px; width:100%;" onclick="location.href='api.php?action=logout'">注销登录</button>
        <?php endif; ?>
    </div>
</div>

<script>
// SVG 雷达图逻辑 (保持极致简洁)
function drawRadar(scores) {
    const keys = ['SOM','OCD','INT','DEP','ANX','HOS','PHOB','PAR','PSY'];
    const labels = ['躯体','强迫','人际','抑郁','焦虑','敌对','恐怖','偏执','精神'];
    const size = 300, center = 150, radius = 100;
    const angleStep = (Math.PI * 2) / keys.length;

    let svg = `<svg width="${size}" height="${size}" viewBox="0 0 ${size} ${size}">`;
    // 绘制 5 层网格
    for(let r=1; r<=5; r++) {
        let pts = keys.map((_, i) => {
            let cr = (radius/5)*r;
            return `${center+cr*Math.cos(i*angleStep-Math.PI/2)},${center+cr*Math.sin(i*angleStep-Math.PI/2)}`;
        });
        svg += `<polygon points="${pts.join(' ')}" fill="none" stroke="#E5E5EA" />`;
    }
    // 绘制得分区
    let dpts = keys.map((k, i) => {
        let val = Math.min(5, parseFloat(scores[k]));
        let cr = (radius/5)*val;
        return `${center+cr*Math.cos(i*angleStep-Math.PI/2)},${center+cr*Math.sin(i*angleStep-Math.PI/2)}`;
    });
    svg += `<polygon points="${dpts.join(' ')}" fill="rgba(0,122,255,0.15)" stroke="var(--ios-blue)" stroke-width="2" />`;
    // 绘制标签
    keys.forEach((k, i) => {
        let x = center + (radius+25)*Math.cos(i*angleStep-Math.PI/2);
        let y = center + (radius+25)*Math.sin(i*angleStep-Math.PI/2);
        svg += `<text x="${x}" y="${y}" text-anchor="middle" dominant-baseline="middle" font-size="11" fill="#8E8E93">${labels[i]}</text>`;
    });
    document.getElementById('radarChart').innerHTML = svg + `</svg>`;
}
drawRadar(<?php echo $report['score_json']; ?>);
</script>
</body>
</html>